import { makeWASocket, makeCacheableSignalKeyStore, DisconnectReason, isJidBroadcast, isJidNewsletter, isJidStatusBroadcast, fetchLatestWaWebVersion, isJidBot, isJidMetaIa, } from 'baileys';
import { consola } from 'consola';
import pino from 'pino';
import QRCode from 'qrcode';
import NodeCache from 'node-cache';
import useAdonisAuthState from '#wa/utils/use_adonis_usestate';
import events from '#wa/event/index';
import DeviceAuth from '#models/device_auth';
import { DateTime } from 'luxon';
const MAX_RECONNECT_ATTEMPTS = 10;
const VERSION_CACHE_DURATION = 3600000;
const CONNECTION_TIMEOUT = 40000;
const PAIRING_CODE_DELAY = 5000;
const SESSIONS = new Map();
const SESSIONS_STATUS = new Map();
const SESSIONS_MAX_RECONNECTS = new Map();
const TIMEOUTS = new Map();
const PAIRING_TIMEOUTS = new Map();
const INIT_LOCKS = new Map();
const WSOCKET = new Map();
const GROUP_CACHE = new NodeCache({
    stdTTL: 300,
    checkperiod: 320,
    maxKeys: 1000,
    useClones: false,
});
let cachedWAVersion = null;
let versionCacheTime = 0;
const getWAVersion = async () => {
    const now = DateTime.now().toMillis();
    if (!cachedWAVersion || now - versionCacheTime > VERSION_CACHE_DURATION) {
        consola.info('[WA] Fetching latest WA Web version...');
        cachedWAVersion = await fetchLatestWaWebVersion({});
        versionCacheTime = now;
        consola.success(`[WA] Version cached: ${cachedWAVersion.version.join('.')}`);
    }
    return cachedWAVersion;
};
const clearAllTimeouts = (id) => {
    if (TIMEOUTS.has(id)) {
        clearTimeout(TIMEOUTS.get(id));
        TIMEOUTS.delete(id);
    }
    if (PAIRING_TIMEOUTS.has(id)) {
        clearTimeout(PAIRING_TIMEOUTS.get(id));
        PAIRING_TIMEOUTS.delete(id);
    }
};
const setConnectionTimeout = (id) => {
    clearAllTimeouts(id);
    const timeoutId = setTimeout(async () => {
        consola.warn(`[WA: ${id}] Connection timeout (${CONNECTION_TIMEOUT}ms). Destroying session...`);
        emitWSocket(id, 'device:status', {
            id,
            type: 'connection:timeout',
            message: 'Connection timeout. Session destroyed automatically.',
        });
        const logout = await removeWASession(id);
        consola.info(`[WA: ${id}] ${logout.message}`);
    }, CONNECTION_TIMEOUT);
    TIMEOUTS.set(id, timeoutId);
    consola.info(`[WA: ${id}] Connection timeout set for ${CONNECTION_TIMEOUT}ms`);
};
const emitWSocket = (id, key, data) => {
    const sockets = WSOCKET.get(id);
    if (sockets && sockets.size > 0) {
        sockets.forEach((s) => {
            try {
                s.emit(key, data);
            }
            catch (error) {
                consola.error(`[WA: ${id}] Error emitting to socket: ${error.message}`);
            }
        });
    }
};
export const getWASession = (id) => {
    return {
        status: SESSIONS_STATUS.get(id) || 'disconnected',
        sock: SESSIONS.get(id) || null,
    };
};
export const healthCheckSession = (id) => {
    const sock = SESSIONS.get(id);
    const status = SESSIONS_STATUS.get(id);
    return {
        exists: !!sock,
        status,
        reconnectAttempts: SESSIONS_MAX_RECONNECTS.get(id) || 0,
        hasTimeout: TIMEOUTS.has(id),
        hasPairingTimeout: PAIRING_TIMEOUTS.has(id),
        isLocked: INIT_LOCKS.has(id),
    };
};
export const initWASession = async ({ id, usePairingCode = false, reconnecting = false, }) => {
    if (INIT_LOCKS.has(id)) {
        consola.warn(`[WA: ${id}] Session initialization already in progress`);
        return null;
    }
    if (SESSIONS.has(id) && !reconnecting) {
        consola.info(`[WA: ${id}] Session already exists, returning existing socket`);
        return SESSIONS.get(id);
    }
    INIT_LOCKS.set(id, true);
    try {
        if (reconnecting) {
            const currentAttempts = (SESSIONS_MAX_RECONNECTS.get(id) || 0) + 1;
            SESSIONS_MAX_RECONNECTS.set(id, currentAttempts);
            if (currentAttempts > MAX_RECONNECT_ATTEMPTS) {
                consola.warn(`[WA: ${id}] Max reconnect attempts reached (${MAX_RECONNECT_ATTEMPTS}). Stopping session!`);
                const result = await stopWASession(id);
                consola.info(`[WA: ${id}] ${result.message}`);
                emitWSocket(id, 'device:status', {
                    id,
                    type: 'connection:max_reconnect_reached',
                    message: `Max reconnect attempts reached (${MAX_RECONNECT_ATTEMPTS}). Session stopped.`,
                });
                return null;
            }
            const oldSock = SESSIONS.get(id);
            if (oldSock) {
                try {
                    oldSock.ev?.removeAllListeners();
                    oldSock.end?.();
                }
                catch (error) {
                    consola.warn(`[WA: ${id}] Error cleaning old socket: ${error.message}`);
                }
            }
            clearAllTimeouts(id);
        }
        const deviceId = id.split('-')[0] || null;
        const deviceNumber = id.split('-')[1] || null;
        if (!deviceId) {
            consola.error(`[WA: ${id}] Invalid device ID`);
            return null;
        }
        const { state, saveCreds } = await useAdonisAuthState({ session: id });
        const { version } = await getWAVersion();
        const sock = makeWASocket({
            version,
            auth: {
                creds: state.creds,
                keys: makeCacheableSignalKeyStore(state.keys, pino({ level: 'silent' })),
            },
            logger: pino({ level: 'silent' }),
            markOnlineOnConnect: true,
            generateHighQualityLinkPreview: true,
            syncFullHistory: false,
            defaultQueryTimeoutMs: undefined,
            shouldIgnoreJid: (jid) => isJidBroadcast(jid) ||
                isJidNewsletter(jid) ||
                isJidStatusBroadcast(jid) ||
                isJidBot(jid) ||
                isJidMetaIa(jid),
        });
        SESSIONS.set(id, sock);
        SESSIONS_STATUS.set(id, 'initializing');
        sock.setGroupCache = (groupId, metadata) => {
            const cacheKey = `${id}:${groupId}`;
            GROUP_CACHE.set(cacheKey, metadata);
        };
        sock.getGroupCache = async (groupId) => {
            const cacheKey = `${id}:${groupId}`;
            if (GROUP_CACHE.has(cacheKey)) {
                return GROUP_CACHE.get(cacheKey);
            }
            try {
                const metadata = await sock.groupMetadata(groupId);
                GROUP_CACHE.set(cacheKey, metadata);
                return metadata;
            }
            catch (error) {
                consola.error(`[WA: ${id}] Error fetching group metadata: ${error.message}`);
                return undefined;
            }
        };
        sock.delGroupCache = (groupId) => {
            const cacheKey = `${id}:${groupId}`;
            GROUP_CACHE.del(cacheKey);
        };
        sock.ev.on('creds.update', saveCreds);
        if (usePairingCode && !sock.authState.creds.registered && deviceNumber) {
            const pairingTimeoutId = setTimeout(async () => {
                try {
                    const code = await sock.requestPairingCode(deviceNumber);
                    emitWSocket(id, 'device:status', {
                        id,
                        type: 'connection:pairing',
                        message: 'Pairing Code Received',
                        data: {
                            code,
                            timeout: CONNECTION_TIMEOUT,
                        },
                    });
                    consola.success(`[WA: ${id}] Pairing Code: ${code}`);
                    setConnectionTimeout(id);
                }
                catch (error) {
                    consola.error(`[WA: ${id}] Error requesting pairing code: ${error.message}`);
                    emitWSocket(id, 'device:status', {
                        id,
                        type: 'connection:pairing_error',
                        message: 'Error requesting pairing code',
                    });
                }
                finally {
                    PAIRING_TIMEOUTS.delete(id);
                }
            }, PAIRING_CODE_DELAY);
            PAIRING_TIMEOUTS.set(id, pairingTimeoutId);
        }
        sock.ev.on('connection.update', async (update) => {
            try {
                const { connection, lastDisconnect, qr } = update;
                if (qr && usePairingCode === false) {
                    const qrDataURL = await QRCode.toDataURL(qr);
                    emitWSocket(id, 'device:status', {
                        id,
                        type: 'connection:qr',
                        message: 'QR Code Received',
                        data: {
                            qr: qrDataURL,
                            timeout: CONNECTION_TIMEOUT,
                        },
                    });
                    consola.success(`[WA: ${id}] QR Code Received`);
                    setConnectionTimeout(id);
                }
                if (connection === 'close') {
                    const shouldReconnect = lastDisconnect?.error?.output?.statusCode !== DisconnectReason.loggedOut;
                    if (shouldReconnect) {
                        SESSIONS_STATUS.set(id, 'reconnecting');
                        consola.warn(`[WA: ${id}] Connection closed, attempting reconnect...`);
                        emitWSocket(id, 'device:status', {
                            id,
                            type: 'connection:reconnecting',
                            message: 'Reconnecting...',
                        });
                        setTimeout(() => {
                            initWASession({
                                id,
                                usePairingCode,
                                reconnecting: true,
                            });
                        }, 2000);
                    }
                    else {
                        SESSIONS_STATUS.set(id, 'disconnected');
                        consola.info(`[WA: ${id}] Logged out, removing session`);
                        const logout = await removeWASession(id);
                        emitWSocket(id, 'device:status', {
                            id,
                            type: 'connection:logout',
                            message: logout.message,
                        });
                    }
                }
                else if (connection === 'connecting') {
                    SESSIONS_STATUS.set(id, 'connecting');
                    consola.info(`[WA: ${id}] Connecting...`);
                    emitWSocket(id, 'device:status', {
                        id,
                        type: 'connection:connecting',
                        message: 'Connecting...',
                    });
                }
                else if (connection === 'open') {
                    SESSIONS_STATUS.set(id, 'connected');
                    consola.success(`[WA: ${id}] Connection Opened`);
                    emitWSocket(id, 'device:status', {
                        id,
                        type: 'connection:open',
                        message: 'Connected',
                    });
                    SESSIONS_MAX_RECONNECTS.delete(id);
                    clearAllTimeouts(id);
                }
            }
            catch (error) {
                consola.error(`[WA: ${id}] Connection Update Error: ${error.message}`);
            }
        });
        await events({ id, sock });
        return sock;
    }
    catch (error) {
        consola.error(`[WA: ${id}] Error initializing session: ${error.message}`);
        return null;
    }
    finally {
        INIT_LOCKS.delete(id);
    }
};
export const stopWASession = async (id) => {
    const sock = SESSIONS.get(id);
    if (!sock) {
        return {
            ok: false,
            message: 'Session not found',
        };
    }
    try {
        sock.ev?.removeAllListeners();
        sock.end?.();
        SESSIONS.delete(id);
        SESSIONS_STATUS.set(id, 'stopped');
        SESSIONS_MAX_RECONNECTS.delete(id);
        clearAllTimeouts(id);
        const cacheKeys = GROUP_CACHE.keys().filter((key) => key.startsWith(`${id}:`));
        cacheKeys.forEach((key) => GROUP_CACHE.del(key));
        return {
            ok: true,
            message: 'Session stopped successfully',
        };
    }
    catch (error) {
        consola.error(`[WA: ${id}] Error stopping session: ${error.message}`);
        return {
            ok: false,
            message: error.message,
        };
    }
};
export const removeWASession = async (id) => {
    const sock = SESSIONS.get(id);
    if (!sock) {
        return {
            ok: false,
            message: 'Session not found',
        };
    }
    try {
        sock.ev?.removeAllListeners();
        const { removeCreds } = await useAdonisAuthState({ session: id });
        removeCreds();
        SESSIONS.delete(id);
        SESSIONS_STATUS.delete(id);
        SESSIONS_MAX_RECONNECTS.delete(id);
        clearAllTimeouts(id);
        const cacheKeys = GROUP_CACHE.keys().filter((key) => key.startsWith(`${id}:`));
        cacheKeys.forEach((key) => GROUP_CACHE.del(key));
        try {
            await sock.logout();
        }
        catch (logoutError) {
            consola.warn(`[WA: ${id}] Logout error (common if connection already closed): ${logoutError.message}`);
        }
        sock.end?.();
        return {
            ok: true,
            message: 'Session destroyed successfully',
        };
    }
    catch (error) {
        consola.error(`[WA: ${id}] Error removing session:`, error);
        return {
            ok: false,
            message: error.message,
        };
    }
};
export const manageAutoStart = async () => {
    try {
        const sessions = await DeviceAuth.query()
            .select('session')
            .distinct('session')
            .whereNotNull('session');
        if (sessions.length === 0) {
            return {
                length: 0,
                succeeded: 0,
                failed: 0,
                message: 'No sessions to auto-start',
            };
        }
        const results = await Promise.allSettled(sessions.map((s) => initWASession({ id: s.session })));
        const succeeded = results.filter((r) => r.status === 'fulfilled').length;
        const failed = results.filter((r) => r.status === 'rejected').length;
        return {
            length: sessions.length,
            succeeded,
            failed,
            message: `Auto-start completed: ${succeeded} succeeded, ${failed} failed`,
        };
    }
    catch (error) {
        consola.error(`[WA] Error in auto-start: ${error.message}`);
    }
};
export const shutdownAllSessions = async () => {
    const sessionIds = Array.from(SESSIONS.keys());
    const results = await Promise.allSettled(sessionIds.map((id) => stopWASession(id)));
    TIMEOUTS.forEach((t) => clearTimeout(t));
    TIMEOUTS.clear();
    PAIRING_TIMEOUTS.forEach((t) => clearTimeout(t));
    PAIRING_TIMEOUTS.clear();
    INIT_LOCKS.clear();
    GROUP_CACHE.flushAll();
    const succeeded = results.filter((r) => r.status === 'fulfilled').length;
    consola.success(`[WA] Shutdown completed: ${succeeded}/${sessionIds.length} sessions stopped`);
};
export { WSOCKET, SESSIONS, SESSIONS_STATUS };
//# sourceMappingURL=whatsapp.js.map